/************************************************************************************************************\

Module Name:    LMoes.h

Description:    Liberatus for Matrox OpenGL ES module.

References:     None.

    Copyright (c) 2011, Matrox Graphics Inc.
    All Rights Reserved.

\************************************************************************************************************/
#ifndef LMOES_H_
#define LMOES_H_

// -----------------------------------------------------------------------------------------------------------
//                                 I N C L U D E S   A N D   U S I N G S
// -----------------------------------------------------------------------------------------------------------
#include "Liberatus.h"

// -----------------------------------------------------------------------------------------------------------
//                                 C O N S T A N T S   A N D   T Y P E S
// -----------------------------------------------------------------------------------------------------------
typedef struct LMOES_OBJECT*         LMoes_Handle;

/************************************************************************************************************\

Enum:           LMoes_AttributeType

Description:    Enumeration of the types of attributes supported by LMoes.

Comments:

\************************************************************************************************************/
typedef enum
{
    LMoes_AttributeType_NONE = 0,
    LMoes_AttributeType_CONTEXT,
    LMoes_AttributeType_COLORBUFFER,
    LMoes_AttributeType_SWAPBUFFERS
} LMoes_AttributeType;

/************************************************************************************************************\

Structure:      LMoes_ContextAttributes

Description:    Structure for rendering context creation attributes.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLMoes_ContextAttributes
{
    LMoes_AttributeType  eAttributeType;           // must be _CONTEXT
    MUINT8               uiMajorVersion;           // default 1
    MUINT8               uiMinorVersion;           // default 0
    MUINT32              bColorBuffer         : 1; // default TRUE
    MUINT32              bDepthStencilBuffer  : 1; // default FALSE
    MUINT32              uiWidth;                  // default 1
    MUINT32              uiHeight;                 // default 1
    LPixelFormat         ePixelFormat;             // default RGB32
    LDeviceThread_Handle hDeviceThread;            // default NULL
} LMoes_ContextAttributes;

/************************************************************************************************************\

Structure:      LMoes_ColorBufferAttributes

Description:    Structure for color buffer attributes.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLMoes_ColorBufferAttributes
{
    LMoes_AttributeType  eAttributeType;  // must be _COLORBUFFER
    LBuffer_Handle       hDrawBuffer;
    LBuffer_Handle       hReadBuffer;
} LMoes_ColorBufferAttributes;

/************************************************************************************************************\

Enum:           LMoes_SwapBuffersMode

Description:    Enumeration of SwapBuffers modes.

Comments:

\************************************************************************************************************/
typedef enum
{
    LMoes_SwapBuffersMode_COPY
} LMoes_SwapBuffersMode;

/************************************************************************************************************\

Structure:      LMoes_SwapBufferAttributes

Description:    Structure for swap buffer attributes.

Comments:       None.

\************************************************************************************************************/
typedef struct tagLMoes_SwapBuffersAttributes
{
    LMoes_AttributeType   eAttributeType;          // must be _SWAPBUFFERS
    LMoes_SwapBuffersMode eMode;                   // default SWAP
    LBuffer_Handle        hDesktop;                // default NULL
    MUINT32               uiWindowX;               // default 0
    MUINT32               uiWindowY;               // default 0
    MUINT32               uiWindowWidth;           // default 0
    MUINT32               uiWindowHeight;          // default 0
    MUINT64               uiWindowTracker;         // default NULL
    MFLAG32               uiFlags;                 // default 0
} LMoes_SwapBuffersAttributes;

/************************************************************************************************************\

Enum:           LMoes_TexImageBufferMode

Description:    Enumeration of TexImageBuffer modes.

Comments:

\************************************************************************************************************/
typedef enum
{
    LMoes_TexImageBufferMode_COPY,
    LMoes_TexImageBufferMode_AUTOMIPMAP,
    LMoes_TexImageBufferMode_BIND
} LMoes_TexImageBufferMode;

// -----------------------------------------------------------------------------------------------------------
//                           G L O B A L   V A R I A B L E   R E F E R E N C E S
// -----------------------------------------------------------------------------------------------------------

// -----------------------------------------------------------------------------------------------------------
//                         G L O B A L   F U N C T I O N   D E C L A R A T I O N S
// -----------------------------------------------------------------------------------------------------------

#if defined (__cplusplus)
extern "C" {
#endif

/************************************************************************************************************\

Function:       LMoes_Create

Description:    Create an object of the LMoes class representing an OpenGL ES rendering context. We further use
                the -"Context" terminology to refer to this object.

Parameters:     IN  hDevice         Handle of the device for which to create a rendering context.
                IN  peAttributes    Pointer to the attribute type member of a structure of type
                                    LMoes_ContextAttributes describing the attributes of the context to
                                    create. A NULL pointer indicates the use of the default attributes.
                OUT phContext       Pointer to a handle to receive the handle of the created rendering
                                    context.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       The method receives as input a set of immutable attributes for the rendering context. The
                structure shall be of type LMoes_AttributeType_CONTEXT. It is possible to pass a NULL pointer
                to the attributes structure indicating that the default values shall be used.

                The maximum width and height of the color and pixmap buffers can be retrieved using the
                LMoes_GetAttributes function with the LMoes_AttributeType_CONFIG type.

                The following pixel formats types are available: RGB32, ARGB32, RGB565, RGB555, ARGB555,
                GRAY8.

\************************************************************************************************************/
LAPI LStatus LMoes_Create
(
    LDevice_Handle       hDevice,
    LMoes_AttributeType* peAttributes,
    LMoes_Handle*        phContext
);

/************************************************************************************************************\

Function:       LMoes_Destroy

Description:    Destroy an LMoes rendering context.

Parameters:     IN hContext     Handle of rendering context.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       None.

\************************************************************************************************************/
LAPI LStatus LMoes_Destroy
(
    LMoes_Handle hContext
);

/************************************************************************************************************\

Function:       LMoes_GetAttributes

Description:    This method provide a mechanism to retrieve attributes from a rendering context. The type of
                attributes is provided by the output structure.

Parameters:     IN     hContext     Handle of a rendering context.
                IN OUT peAttributes Pointer to the attribute type member of a structure of attributes. On
                                    input the type member describes the type of the structure and on output
                                    the remaining members of the structure are filled with the attributes
                                    information.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       Only one type of attributes can be retrieved at a time.

\************************************************************************************************************/
LAPI LStatus LMoes_GetAttributes
(
    LMoes_Handle         hContext,
    LMoes_AttributeType* peAttributes
);

/************************************************************************************************************\

Function:       LMoes_GetDefaultAttributes

Description:    This method provide a mechanism to retrieve the default attributes from a rendering context.
                The type of attributes is provided by the output structure.

Parameters:     IN     hContext     Handle of a rendering context.
                IN OUT peAttributes Pointer to the attribute type member of a structure of attributes. On
                                    input the type member describes the type of the structure and on output
                                    the remaining members of the structure are filled with the attributes
                                    information.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       Only one type of attributes can be retrieved at a time. This function provides a mechanism
                for retrieving an attribute structure filled with default values such that an application
                only have to care about the subset of the members it is interested in.

\************************************************************************************************************/
LAPI LStatus LMoes_GetDefaultAttributes
(
    LMoes_Handle         hContext,
    LMoes_AttributeType* peAttributes
);

/************************************************************************************************************\

Function:       LMoes_MakeCurrentContext

Description:    Make a rendering context current to the calling CPU thread.

Parameters:     IN     hContext     Handle of a rendering context.
                IN     hDrawBuffer  Handle of a compatible buffer to use as the draw buffer
                                    or NULL to use the default color buffer.
                IN     hReadBuffer  Handle of a compatible buffer to use as the read buffer
                                    or NULL to use the default color buffer.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.
                LStatus_UNSUPPORTED     Buffer metrics or pixel format are not supported.

Comments:       This function binds the rendering context with the current CPU thread and optionally
                binds compatible external draw and read buffers to the rendering context. Passing a
                NULL external buffer handle to the function indicates the use of the default color buffer.
                The following pixel format types are supported for the draw and read buffers: RGB32, ARGB32,
                RGB565, RGB555, ARGB555, GRAY8.

\************************************************************************************************************/
LAPI LStatus LMoes_MakeCurrentContext
(
    LMoes_Handle   hContext,
    LBuffer_Handle hDrawBuffer,
    LBuffer_Handle hReadBuffer
);

/************************************************************************************************************\

Function:       LMoes_GetCurrentContext

Description:    Retrieve the current rendering context handle of the calling CPU thread and the associated
                current draw and read buffer handles.

Parameters:     OUT    phContext     Pointer to a handle to receive the current rendering context handle or
                                     NULL if this information is not requested.
                OUT    phDrawBuffer  Pointer to handle to receive the buffer to the current draw buffer handle
                                     or NULL if this information is not requested.
                OUT    phReadBuffer  Pointer to a handle to receive the current read buffer handle or NULL if
                                     this information is not requested.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       If NULL is passed to a call to LMoes_MakeCurrentContext as the draw buffer handle, a subsequent
                call to LMoes_GetCurrentContext will not return NULL as the draw buffer. Instead the function
                will return a handle to an internal buffer.

\************************************************************************************************************/
LAPI LStatus LMoes_GetCurrentContext
(
    LMoes_Handle*   phContext,
    LBuffer_Handle* phDrawBuffer,
    LBuffer_Handle* phReadBuffer
);

/************************************************************************************************************\

Function:       LMoes_GetProcAddress

Description:    This method is used to get the entry-points of OpenGL ES extensions.

Parameters:     IN     hContext         Handle of a rendering context.

                IN     szProcName       Name of the extension function.

                OUT    ppfnProcAddress  Pointer to an extension function pointer.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.
                LStatus_UNSUPPORTED     The extension related to the function name is not supported.

Comments:

\************************************************************************************************************/
LAPI LStatus LMoes_GetProcAddress
(
    LMoes_Handle hContext,
    char*        szProcName,
    void (**ppfnProcAddress)(void)
);

/************************************************************************************************************\

Function:       LMoes_SwapBuffers

Description:    This function publishes the content of the current draw buffer to the desktop buffer
                according to the specified mode of operation and attributes.

                The possible modes are:

                    LMoes_SwapBuffersMode_COPY

                    The calling CPU thread is put to sleep while waiting for the previous LMoes_SwapBuffers
                    to complete. Then the current component color buffer is copied to the desktop buffer.

Parameters:     IN hContext     Handle of a rendering context.
                IN peAttributes Pointer to the attribute type member of a structure of type
                                LMoes_SwapBuffersAttributes describing the attributes of the swap buffers
                                operation. A NULL pointer indicates the use of the default attributes.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       The COPY mode keeps the content of the draw buffer valid.

                The swap-buffer operation can be synchronized with the display retrace or any other event
                issuing a synchronization command in the same device-thread prior to issuing the swap-buffer
                operation.

                The window region tracking functionality is specific to the operating system implementation.
                The window tracking object is totally opaque to the LMoes module that forward the handle to
                the kernel driver along with the swap-buffer operation. In the operating system kernel the
                tracking object provides the effective desktop buffer and region of a window for the
                swap-buffer operation.

\************************************************************************************************************/
LAPI LStatus LMoes_SwapBuffers
(
    LMoes_Handle         hContext,
    LMoes_AttributeType* peAttributes
);

/************************************************************************************************************\

Function:       LStatus LMoes_TexImageBuffer

Description:    Provide texture images to a texture object from a buffer

Parameters:     IN  hContext        Handle of a rendering context.
                IN  uiTexture       Name of a texture object.
                IN  uiTarget        Target texture image.
                IN  iLevel          Level of texture image mipmap.
                IN  iLayer          Layer of texture image layer. Must be 0.
                IN  iInternalFormat Internal format of texture image.
                IN  eMode           Mode of operation.
                IN  hBuffer         Handle of a compatible buffer.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.
                LStatus_UNSUPPORTED     The current operation is not supported.

Comments:       Mode of operation:
                LMoes_TexImageBufferMode_COPY        Copy the content of the buffer to the selected level and
                                                     layer of the texture image.
                LMoes_TexImageBufferMode_AUTOMIPMAP  Copy the content of the buffer to the level 0 of the
                                                     selected layer of the texture image and automatically
                                                     generate the subsequent mipmap levels.
                LMoes_TexImageBufferMode_BIND        Bind the buffer to the level 0 and layer 0 of the texture
                                                     image.

                The COPY mode supports a limited form of format conversion. The internal format parameter can
                be any of the base internal formats described in table 3.3 of the OpenGL ES 1.0 specification,
                subject to the constraint that buffer components can be dropped during the conversion to the
                base internal format, but new components cannot be added. The error LStatus_UNSUPPORTED is
                returned when the format conversion is not possible.
                [todo] Issue with components ordering of G450 not matching those of ES.

                The AUTOMIPMAP mode must be called with iLevel and iLayer equal to 0. It operates as the COPY
                mode for the level 0 before using the resulting level 0 image to generate the remaining mipmap
                levels.

                The BIND mode supports the following  pixel format types: RGB32, ARGB32, RGB565, RGB555, ARGB555,
                GRAY8. These parameters restrictions apply: uiTarget must be GL_TEXTURE_2D, iLevel and iLayer
                must be 0. Modifications to the content of the buffer are immediately visible to the texture
                object. The error LStatus_UNSUPPORTED  is returned if the texture object has an external or
                internal buffer bounded.
                [todo] Discuss that only layer 0 of a Liberatus buffer is accessible.

\************************************************************************************************************/
LAPI LStatus LMoes_TexImageBuffer
(
    LMoes_Handle             hContext,
    MUINT                    uiTexture,        // OpenGL texture
    MUINT                    uiTarget,         // OpenGL target
    MINT                     iLevel,           // OpenGL level
    MINT                     iLayer,           // OpenGL layer
    MINT                     iInternalFormat,  // OpenGL internalformat
    LMoes_TexImageBufferMode eMode,
    LBuffer_Handle           hBuffer
);

/************************************************************************************************************\

Function:       LStatus LMoes_GetTexImageBuffer

Description:    Retrieve the buffer providing the texture images of a texture object.

Parameters:     IN  hContext        Handle of a rendering context.
                IN  uiTexture       Name of a texture object.
                IN  uiTarget        Target texture image.
                IN  iLevel          Level of texture image mipmap.
                IN  iLayer          Layer of texture image layer. Must be 0.
                OUT phBuffer        Pointer to a handle to receive the buffer handle.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       A texture object may be in the following states: no texture image, texture image bounded to
                an internal buffer, texture image bounded to an external buffer. This function retrieve the
                current binding of the texture image. NULL indicates no current binding.

                The buffer handle returned by this function can be used as the draw buffer parameter of
                MakeCurrentContext(). That allows rendering to a texture image.

\************************************************************************************************************/
LAPI LStatus LMoes_GetTexImageBuffer
(
    LMoes_Handle    hContext,
    MUINT           uiTexture,      // OpenGL texture
    MUINT           uiTarget,       // OpenGL target
    MINT            iLevel,         // OpenGL level
    MINT            iLayer,         // OpenGL layer
    LBuffer_Handle* phBuffer
);

/************************************************************************************************************\

Function:       LMoes_MarkSync

Description:    This function can be used to synchronize with the execution of OpenGL ES commands. It retrieves
                a synchronization marker indicating the current position in the rendering context time-line.

Parameters:     IN  hContext    Handle of a rendering context.
                OUT puiMarker   Pointer to a 64 bit synchronization marker.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.

Comments:       Each rendering context has a specific time-line.

\************************************************************************************************************/
LAPI LStatus LMoes_MarkSync
(
    LMoes_Handle hContext,
    MUINT64*     puiMarker
);

/************************************************************************************************************\

Function:       LMoes_WaitSync

Description:    This function can be used to synchronize with the execution of OpenGL ES commands. It puts the
                calling CPU thread to sleep until the graphics engine terminates the execution of the
                commands anterior to the synchronization marker and that the buffers' memory is updated.

Parameters:     IN hContext                 Handle of a rendering context.
                IN uiMarker                 Synchronization marker.
                IN uiTimeOutInMilliSeconds  Maximum wait time. Set to LINFINITE_TIMEOUT to disable timeout.

Return Value:   LStatus_OK              The function completed successfully.
                LStatus_FAIL            The function completed with a generic failure.
                LStatus_INVALID_PARAM   An input parameter is invalid.
                LStatus_TIMEOUT         The time specified has passed.

Comments:       This function flushes previous commands to the rendering engine prior to wait for the
                synchronization marker.

\************************************************************************************************************/
LAPI LStatus LMoes_WaitSync
(
    LMoes_Handle hContext,
    MUINT64      uiMarker,
    MUINT32      uiTimeOutInMilliSeconds
);

#if defined (__cplusplus)
}
#endif

#endif /* LMOES_H_ */
